<?php
require_once __DIR__ . "/../../config.php";

function app_log($message) {
  $line = sprintf("[%s] %s\n", date("c"), $message);
  file_put_contents(APP_LOG_FILE, $line, FILE_APPEND);
}

function h($s) { return htmlspecialchars($s ?? "", ENT_QUOTES, 'UTF-8'); }

function require_post($keys) {
  foreach ($keys as $k) {
    if (!isset($_POST[$k]) || $_POST[$k] === "") {
      http_response_code(422);
      echo "Missing field: " . h($k);
      exit;
    }
  }
}

function printful_request($endpoint, $method="GET", $data=null) {
  $url = rtrim(PRINTFUL_API_BASE, "/") . $endpoint;
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Authorization: Bearer " . PRINTFUL_API_KEY,
    "Content-Type: application/json"
  ]);
  if ($method === "POST") {
    curl_setopt($ch, CURLOPT_POST, true);
    if (!is_null($data)) curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
  } elseif ($method === "DELETE") {
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "DELETE");
  }
  $resp = curl_exec($ch);
  if ($resp === false) {
    app_log("cURL error: " . curl_error($ch));
    curl_close($ch);
    return null;
  }
  curl_close($ch);
  return json_decode($resp, true);
}

// Stripe minimal HTTP (no SDK for simplicity)
function stripe_request($path, $fields) {
  $url = "https://api.stripe.com" . $path;
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_USERPWD, STRIPE_SECRET_KEY . ":");
  curl_setopt($ch, CURLOPT_POST, true);
  curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($fields));
  $resp = curl_exec($ch);
  if ($resp === false) {
    app_log("Stripe cURL error: " . curl_error($ch));
    curl_close($ch);
    return null;
  }
  curl_close($ch);
  return json_decode($resp, true);
}

function verify_stripe_sig($payload, $header, $secret) {
  // Minimal verification. In production, use Stripe's official library.
  $parts = [];
  foreach (explode(",", $header) as $kv) {
    $pair = array_map("trim", explode("=", $kv, 2));
    if (count($pair) === 2) $parts[$pair[0]] = $pair[1];
  }
  if (!isset($parts["t"]) || !isset($parts["v1"])) return false;
  $signed_payload = $parts["t"] . "." . $payload;
  $computed = hash_hmac("sha256", $signed_payload, $secret);
  return hash_equals($computed, $parts["v1"]);
}

// PayPal helpers
function paypal_access_token() {
  $ch = curl_init(PAYPAL_API_BASE . "/v1/oauth2/token");
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_USERPWD, PAYPAL_CLIENT_ID . ":" . PAYPAL_CLIENT_SECRET);
  curl_setopt($ch, CURLOPT_POST, true);
  curl_setopt($ch, CURLOPT_POSTFIELDS, "grant_type=client_credentials");
  $resp = curl_exec($ch);
  if ($resp === false) return null;
  curl_close($ch);
  $data = json_decode($resp, true);
  return $data["access_token"] ?? null;
}

function paypal_request($method, $path, $data=null, $token=null) {
  $ch = curl_init(PAYPAL_API_BASE . $path);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Authorization: Bearer " . $token,
    "Content-Type: application/json"
  ]);
  if ($method === "POST") {
    curl_setopt($ch, CURLOPT_POST, true);
    if (!is_null($data)) curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
  }
  $resp = curl_exec($ch);
  if ($resp === false) return null;
  curl_close($ch);
  return json_decode($resp, true);
}

// Create & confirm a Printful order
function place_printful_order($customer, $item, $sizeMap) {
  // $customer: [name,address1,city,state_code,country_code,zip,email]
  // $item: [title,image_url,size,quantity]
  $size = $item["size"];
  if (!isset($sizeMap[$size])) {
    throw new Exception("Unknown size variant: " . $size);
  }
  $variant_id = $sizeMap[$size];

  $payload = [
    "external_id" => "@ORDER-" . time() . "-" . rand(1000,9999),
    "recipient" => [
      "name" => $customer["name"],
      "address1" => $customer["address1"],
      "city" => $customer["city"],
      "state_code" => $customer["state_code"],
      "country_code" => $customer["country_code"],
      "zip" => $customer["zip"],
      "email" => $customer["email"]
    ],
    "items" => [[
      "variant_id" => $variant_id,
      "quantity" => intval($item["quantity"] ?? 1),
      "name" => $item["title"],
      "files" => [[ "url" => $item["image_url"] ]]
    ]]
  ];

  $create = printful_request("/orders", "POST", $payload);
  if (!$create || !isset($create["result"]["id"])) {
    app_log("Printful order create failed: " . json_encode($create));
    return null;
  }
  $order_id = $create["result"]["id"];

  // Confirm to start fulfillment
  $confirm = printful_request("/orders/" . $order_id . "/confirm", "POST", []);
  return ["create" => $create, "confirm" => $confirm];
}
